package mednt.dbconn;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;

import mednt.lib.Err;

public class DBConn {

	private final String connStr;
	private final String user;
	private final String pass;
	private final boolean mysql;
	private Connection conn;
	private final boolean debug;

	/**
	 * @param _debug
	 *            jeżeli true to {@link DBConn#query(String)} oraz {@link DBConn#execute(String)} logują w konsoli wszystkie wykonywane zapytania
	 */
	public DBConn(String _connStr, String _user, String _pass, boolean _mysql, boolean _debug) {
		this.connStr = _connStr;
		this.user = _user;
		this.pass = _pass;
		this.mysql = _mysql;
		this.debug = _debug;

		this.conn = null; // lazy connection
	}

	public DBConn(String _connStr, String _user, String _pass) throws ClassNotFoundException, SQLException {
		this(_connStr, _user, _pass, false, false);
	}

	public void close() {
		try {
			if (this.conn != null)
				this.conn.close();
		} catch (SQLException e) {
			Err.logException(e);
		}
	}

	/**
	 * Wykonuje statyczne zapytanie SQL (np. SELECT), ktore zwraca dane
	 * 
	 * @param sql
	 *            statyczne zapytanie SQL
	 * @return SDBResult wyniki zapytania
	 * @throws SQLException
	 */
	public SDBResult query(String sql) {
		if (this.debug)
			System.out.println("Q: " + sql);
		try {
			this.doConnect();
			Statement stmt = this.conn.createStatement();
			ResultSet rs = stmt.executeQuery(sql);
			SDBResult res = new SDBResult(rs);
			rs.close();
			stmt.close();
			return res;
		} catch (SQLException e) {
			throw Err.logAndThrowRuntimeException(e);
		} catch (ClassNotFoundException e) {
			throw Err.logAndThrowRuntimeException(e);
		}
	}

	private void doConnect() throws ClassNotFoundException, SQLException {
		if (this.conn != null)
			return;
		this.conn = DriverManager.getConnection(this.connStr, this.user, this.pass);
	}

	/**
	 * Wykonuje zapytanie DML (np. INSERT, DELETE, UPDATE) lub zapytanie DDL (np. CREATE), które nie zwracają danych
	 * 
	 * @param sql
	 *            zapytanie DML lub DDL
	 * @return int id wstawionego wiersza
	 * @throws SQLException
	 */
	public int execute(String sql) {
		if (this.debug)
			System.out.println("Q: " + sql);

		try {
			this.doConnect();
			Statement stmt = this.conn.createStatement();
			int resCode = stmt.executeUpdate(sql, Statement.RETURN_GENERATED_KEYS);

			try (ResultSet generatedKeys = stmt.getGeneratedKeys()) {
				if (generatedKeys.next())
					resCode = generatedKeys.getInt(1);
			}
			stmt.close();
			return resCode;
		} catch (SQLException e) {
			throw Err.logAndThrowRuntimeException(e);
		} catch (ClassNotFoundException e) {
			throw Err.logAndThrowRuntimeException(e);
		}
	}

	public String single(String sql) {
		SDBResult query = this.query(sql);
		if (query.rows() == 0)
			return "";
		return query.s(0, 0);
	}

	public String escape(String v) {
		if (this.mysql)
			return v.replace("'", "''").replace("\\", "\\\\");
		else
			return v.replace("'", "''");
	}
}
